from django.contrib import admin
from django.contrib.auth.admin import UserAdmin as BaseUserAdmin
from django.contrib.auth.models import Group
from django.http import HttpResponse
import openpyxl
from openpyxl.styles import Font, Alignment, PatternFill, Border, Side
from .forms import UserChangeForm, UserCreateForm
from .models import User
from home.models import Activity

class UserAdmin(BaseUserAdmin):
    form = UserChangeForm
    add_form = UserCreateForm
    list_display = ('username', 'first_name', 'last_name', 'age', 'disease', 'address', 'phone')
    list_filter = ('username', 'is_active')
    search_fields = ('username',)
    ordering = ('username',)
    filter_horizontal = ()
    actions = ['export_to_excel']

    fieldsets = (
        ('User Info', {'fields': ('username', 'password', 'first_name', 'last_name', 'age', 'disease', 'address', 'phone')}),
        ('Permissions', {'fields': ('is_admin', 'is_active')}),
    )

    add_fieldsets = (
        (None, {'fields': ('username', 'first_name', 'last_name', 'age', 'disease', 'address', 'phone', 'password1', 'password2')}),
    )

    def export_to_excel(self, request, queryset):
        wb = openpyxl.Workbook()
        wb.remove(wb.active)  # remove default sheet

        for user in queryset:
            activities = Activity.objects.filter(user=user)
            ws = wb.create_sheet(title=user.username)

            if not activities.exists():
                ws.append(['No activity data found for this user.'])
                continue

            headers = [field.name for field in Activity._meta.fields if field.name != 'user']
            ws.append(headers)

            # Style headers
            header_font = Font(bold=True, color="FFFFFF")
            header_fill = PatternFill("solid", fgColor="4F81BD")
            thin_border = Border(left=Side(style='thin'), right=Side(style='thin'),
                                 top=Side(style='thin'), bottom=Side(style='thin'))

            for col_num, column_title in enumerate(headers, 1):
                cell = ws.cell(row=1, column=col_num)
                cell.font = header_font
                cell.fill = header_fill
                cell.alignment = Alignment(horizontal='center')
                cell.border = thin_border

            for idx, activity in enumerate(activities, start=2):
                row = [getattr(activity, field) for field in headers]
                for col_num, value in enumerate(row, 1):
                    cell = ws.cell(row=idx, column=col_num, value=value)
                    cell.alignment = Alignment(horizontal='center')
                    cell.border = thin_border

            for column_cells in ws.columns:
                max_length = max(len(str(cell.value)) if cell.value is not None else 0 for cell in column_cells)
                adjusted_width = (max_length + 2)
                ws.column_dimensions[column_cells[0].column_letter].width = adjusted_width

        response = HttpResponse(
            content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        )
        response['Content-Disposition'] = 'attachment; filename="users_activity_export.xlsx"'
        wb.save(response)
        return response

    export_to_excel.short_description = " 📤 Export selected users' activities to Excel"

admin.site.register(User, UserAdmin)
admin.site.unregister(Group)
